// Package ops provides a facility for tracking the processing of operations,
// including contextual metadata about the operation and their final success or
// failure. An op is assumed to have succeeded if by the time of calling Exit()
// no errors have been reported. The final status can be reported to a metrics
// facility.
package ops

import (
	gocontext "context"
	"crypto/rand"
	"fmt"
	"sync"
	"time"

	"github.com/getlantern/context"

	"go.opentelemetry.io/otel"
	"go.opentelemetry.io/otel/attribute"
	"go.opentelemetry.io/otel/trace"
)

const (
	_goctx = "_goctx"
)

var (
	cm             = context.NewManager()
	reporters      []Reporter
	reportersMutex sync.RWMutex
	tracer         trace.Tracer
	tracerMx       sync.RWMutex
)

// Reporter is a function that reports the success or failure of an Op. If
// failure is nil, the Op can be considered successful.
type Reporter func(failure error, ctx map[string]interface{})

// Op represents an operation that's being performed. It mimics the API of
// context.Context.
type Op interface {
	// Begin marks the beginning of an Op under this Op.
	Begin(name string) Op

	// Go starts the given function on a new goroutine.
	Go(fn func())

	// End marks the end of this op, at which point the Op will report its success
	// or failure to all registered Reporters.
	End()

	// Cancel cancels this op so that even if End() is called later, it will not
	// report its success or failure.
	Cancel()

	// Set puts a key->value pair into the current Op's context.
	Set(key string, value interface{}) Op

	// SetDynamic puts a key->value pair into the current Op's context, where the
	// value is generated by a function that gets evaluated at every Read.
	SetDynamic(key string, valueFN func() interface{}) Op

	// FailIf marks this Op as failed if the given err is not nil. If FailIf is
	// called multiple times, the latest error will be reported as the failure.
	// Returns the original error for convenient chaining.
	FailIf(err error) error

	// TraceID returns the hex encoded trace ID for the current Op if and only if OpenTelemetry
	// is enabled.
	TraceID() string
}

type op struct {
	ctx       context.Context
	span      trace.Span
	canceled  bool
	failure   error
	failureMx sync.RWMutex
}

// RegisterReporter registers the given reporter.
func RegisterReporter(reporter Reporter) {
	reportersMutex.Lock()
	reporters = append(reporters, reporter)
	reportersMutex.Unlock()
}

// EnableOpenTelemetry enables exporting of traces to OpenTelemetry under the given tracerName.
func EnableOpenTelemetry(tracerName string) {
	tracerMx.Lock()
	defer tracerMx.Unlock()
	tracer = otel.Tracer(tracerName)
}

func getTracer() trace.Tracer {
	tracerMx.RLock()
	defer tracerMx.RUnlock()
	return tracer
}

// Begin marks the beginning of a new Op.
func Begin(name string) Op {
	return addOpenTelemetry(
		name,
		"",
		&op{ctx: cm.Enter().Put("op", name).PutIfAbsent("root_op", name)},
	)
}

func (o *op) Begin(name string) Op {
	return addOpenTelemetry(
		name,
		"",
		&op{ctx: o.ctx.Enter().Put("op", name).PutIfAbsent("root_op", name)},
	)
}

// Resume begins a new op within the scope of an existing trace
func Resume(name, hexTraceID string) Op {
	return addOpenTelemetry(
		name,
		hexTraceID,
		&op{ctx: cm.Enter().Put("op", name).PutIfAbsent("root_op", name)},
	)
}

func (o *op) Resume(name, hexTraceID string) Op {
	return addOpenTelemetry(
		name,
		hexTraceID,
		&op{ctx: o.ctx.Enter().Put("op", name).PutIfAbsent("root_op", name)},
	)
}

func addOpenTelemetry(name, hexTraceID string, o *op) Op {
	t := getTracer()
	if t != nil {
		var parentCtx gocontext.Context
		_parentCtx, hasParentCtx := cm.Get(_goctx)
		if hasParentCtx {
			parentCtx, _ = _parentCtx.(gocontext.Context)
		} else {
			parentCtx = tryResumeTrace(hexTraceID)
		}
		ctx, span := t.Start(parentCtx, name)
		o.span = span
		o.ctx.Put(_goctx, ctx)
	}
	return o
}

func tryResumeTrace(hexTraceID string) gocontext.Context {
	ctx := gocontext.Background()
	if hexTraceID == "" {
		return ctx
	}
	// attempting to resume trace
	traceID, err := trace.TraceIDFromHex(hexTraceID)
	if err != nil {
		fmt.Printf("Unable to resume trace from trace ID %v, will start new trace: %v\n", hexTraceID, err)
		return ctx
	}
	var spanID [8]byte
	_, err = rand.Read(spanID[:])
	if err != nil {
		fmt.Printf("Unable to generate span ID, will start new trace: %v\n", err)
		return ctx
	}
	var spanContextConfig trace.SpanContextConfig
	spanContextConfig.TraceID = traceID
	spanContextConfig.SpanID = spanID
	spanContext := trace.NewSpanContext(spanContextConfig)
	return trace.ContextWithSpanContext(ctx, spanContext)
}

func (o *op) TraceID() string {
	if o.span == nil {
		return ""
	}
	return o.span.SpanContext().TraceID().String()
}

func (o *op) Go(fn func()) {
	o.ctx.Go(fn)
}

// Go mimics the method from context.Manager.
func Go(fn func()) {
	cm.Go(fn)
}

func (o *op) Cancel() {
	o.canceled = true
}

func (o *op) End() {
	if o.canceled {
		return
	}

	var reportersCopy []Reporter
	reportersMutex.RLock()
	if len(reporters) > 0 {
		reportersCopy = make([]Reporter, len(reporters))
		copy(reportersCopy, reporters)
	}
	reportersMutex.RUnlock()

	if len(reportersCopy) > 0 || o.span != nil {
		o.failureMx.RLock()
		failure := o.failure
		o.failureMx.RUnlock()
		ctx := o.ctx.AsMap(failure, true)
		delete(ctx, _goctx)
		if failure != nil {
			_, errorSet := ctx["error"]
			if !errorSet {
				ctx["error"] = failure.Error()
			}
		}

		for _, reporter := range reportersCopy {
			reporter(failure, ctx)
		}

		if o.span != nil {
			for key, _value := range ctx {
				var value attribute.Value
				switch v := _value.(type) {
				case bool:
					value = attribute.BoolValue(v)
				case int:
					value = attribute.Int64Value(int64(v))
				case byte:
					value = attribute.Int64Value(int64(v))
				case int32:
					value = attribute.Int64Value(int64(v))
				case int64:
					value = attribute.Int64Value(int64(v))
				case time.Duration:
					value = attribute.Float64Value(float64(v.Milliseconds()))
				case float32:
					value = attribute.Float64Value(float64(v))
				case float64:
					value = attribute.Float64Value(float64(v))
				case string:
					value = attribute.StringValue(v)
				}
				if value.Type() != attribute.INVALID {
					o.span.SetAttributes(attribute.KeyValue{
						Key:   attribute.Key(key),
						Value: value})
				}
			}
		}
	}

	o.ctx.Exit()

	if o.span != nil {
		o.span.End()
	}
}

func (o *op) Set(key string, value interface{}) Op {
	o.ctx.Put(key, value)
	return o
}

// SetGlobal puts a key->value pair into the global context, which is inherited
// by all Ops.
func SetGlobal(key string, value interface{}) {
	cm.PutGlobal(key, value)
}

func (o *op) SetDynamic(key string, valueFN func() interface{}) Op {
	o.ctx.PutDynamic(key, valueFN)
	return o
}

// SetGlobalDynamic is like SetGlobal but uses a function to derive the value
// at read time.
func SetGlobalDynamic(key string, valueFN func() interface{}) {
	cm.PutGlobalDynamic(key, valueFN)
}

// AsMap mimics the method from context.Manager.
func AsMap(obj interface{}, includeGlobals bool) context.Map {
	result := cm.AsMap(obj, includeGlobals)
	delete(result, _goctx)
	return result
}

func (o *op) FailIf(err error) error {
	if err != nil {
		o.failureMx.Lock()
		o.failure = err
		o.failureMx.Unlock()
		if o.span != nil {
			o.span.RecordError(err)
		}
	}
	return err
}
