package decimal

import (
	"fmt"
	"math/big"
	"sync"
)

// fint (Fast INTeger) is a wrapper around uint64.
type fint uint64

// maxFint is a maximum value of fint.
const maxFint = 9_999_999_999_999_999_999

// pow10 is a cache of powers of 10, where pow10[x] = 10^x.
var pow10 = [...]fint{
	1,                          // 10^0
	10,                         // 10^1
	100,                        // 10^2
	1_000,                      // 10^3
	10_000,                     // 10^4
	100_000,                    // 10^5
	1_000_000,                  // 10^6
	10_000_000,                 // 10^7
	100_000_000,                // 10^8
	1_000_000_000,              // 10^9
	10_000_000_000,             // 10^10
	100_000_000_000,            // 10^11
	1_000_000_000_000,          // 10^12
	10_000_000_000_000,         // 10^13
	100_000_000_000_000,        // 10^14
	1_000_000_000_000_000,      // 10^15
	10_000_000_000_000_000,     // 10^16
	100_000_000_000_000_000,    // 10^17
	1_000_000_000_000_000_000,  // 10^18
	10_000_000_000_000_000_000, // 10^19
}

// add calculates x + y and checks overflow.
func (x fint) add(y fint) (z fint, ok bool) {
	if maxFint-x < y {
		return 0, false
	}
	z = x + y
	return z, true
}

// mul calculates x * y and checks overflow.
func (x fint) mul(y fint) (z fint, ok bool) {
	if y == 0 {
		return 0, true
	}
	z = x * y
	if z/y != x {
		return 0, false
	}
	if z > maxFint {
		return 0, false
	}
	return z, true
}

// quo calculates x / y and checks division by zero and inexact division.
func (x fint) quo(y fint) (z fint, ok bool) {
	if y == 0 {
		return 0, false
	}
	z = x / y
	if z*y != x {
		return 0, false
	}
	return z, true
}

// quoRem calculates q = ⌊x / y⌋, r = x - y * q and checks division by zero.
func (x fint) quoRem(y fint) (q, r fint, ok bool) {
	if y == 0 {
		return 0, 0, false
	}
	q = x / y
	r = x - q*y
	return q, r, true
}

// subAbs calculates |x - y|.
func (x fint) subAbs(y fint) fint {
	if x > y {
		return x - y
	}
	return y - x
}

// lsh (Left Shift) calculates x * 10^shift and checks overflow.
func (x fint) lsh(shift int) (z fint, ok bool) {
	// Special cases
	switch {
	case shift <= 0:
		return x, true
	case shift == 1 && x < maxFint/10: // to speed up common case
		return x * 10, true
	case shift >= len(pow10):
		return 0, false
	}
	// General case
	y := pow10[shift]
	return x.mul(y)
}

// fsa (Fused Shift and Addition) calculates x * 10^shift + b and checks overflow.
func (x fint) fsa(shift int, b byte) (z fint, ok bool) {
	z, ok = x.lsh(shift)
	if !ok {
		return 0, false
	}
	z, ok = z.add(fint(b))
	if !ok {
		return 0, false
	}
	return z, true
}

func (x fint) isOdd() bool {
	return x&1 != 0
}

// rshHalfEven (Right Shift) calculates round(x / 10^shift) and rounds result
// using "half to even" rule.
func (x fint) rshHalfEven(shift int) fint {
	// Special cases
	switch {
	case x == 0:
		return 0
	case shift <= 0:
		return x
	case shift >= len(pow10):
		return 0
	}
	// General case
	y := pow10[shift]
	z := x / y
	r := x - z*y                        // r = x % y
	y = y >> 1                          // y = y / 2, which is safe as y is a multiple of 10
	if y < r || (y == r && z.isOdd()) { // half-to-even
		z++
	}
	return z
}

// rshUp (Right Shift) calculates ⌈x / 10^shift⌉ and rounds result away from zero.
func (x fint) rshUp(shift int) fint {
	// Special cases
	switch {
	case x == 0:
		return 0
	case shift <= 0:
		return x
	case shift >= len(pow10):
		return 1
	}
	// General case
	y := pow10[shift]
	z := x / y
	r := x - z*y // r = x % y
	if r > 0 {
		z++
	}
	return z
}

// rshDown (Right Shift) calculates ⌊x / 10^shift⌋ and rounds result towards zero.
func (x fint) rshDown(shift int) fint {
	// Special cases
	switch {
	case x == 0:
		return 0
	case shift <= 0:
		return x
	case shift >= len(pow10):
		return 0
	}
	// General case
	y := pow10[shift]
	return x / y
}

// prec returns length of x in decimal digits.
// prec assumes that 0 has no digits.
func (x fint) prec() int {
	left, right := 0, len(pow10)
	for left < right {
		mid := (left + right) / 2
		if x < pow10[mid] {
			right = mid
		} else {
			left = mid + 1
		}
	}
	return left
}

// ntz returns number of trailing zeros in x.
// ntz assumes that 0 has no trailing zeros.
func (x fint) ntz() int {
	left, right := 1, x.prec()
	for left < right {
		mid := (left + right) / 2
		if x%pow10[mid] == 0 {
			left = mid + 1
		} else {
			right = mid
		}
	}
	return left - 1
}

// hasPrec returns true if x has given number of digits or more.
// hasPrec assumes that 0 has no digits.
//
// x.hasPrec(p) is significantly faster than x.prec() >= p.
func (x fint) hasPrec(prec int) bool {
	// Special cases
	switch {
	case prec < 1:
		return true
	case prec > len(pow10):
		return false
	}
	// General case
	return x >= pow10[prec-1]
}

// bint (Big INTeger) is a wrapper around big.Int.
type bint big.Int

// bpow10 is a cache of powers of 10, where bpow10[x] = 10^x.
var bpow10 = [...]*bint{
	mustParseBint("1"),
	mustParseBint("10"),
	mustParseBint("100"),
	mustParseBint("1000"),
	mustParseBint("10000"),
	mustParseBint("100000"),
	mustParseBint("1000000"),
	mustParseBint("10000000"),
	mustParseBint("100000000"),
	mustParseBint("1000000000"),
	mustParseBint("10000000000"),
	mustParseBint("100000000000"),
	mustParseBint("1000000000000"),
	mustParseBint("10000000000000"),
	mustParseBint("100000000000000"),
	mustParseBint("1000000000000000"),
	mustParseBint("10000000000000000"),
	mustParseBint("100000000000000000"),
	mustParseBint("1000000000000000000"),
	mustParseBint("10000000000000000000"),
	mustParseBint("100000000000000000000"),
	mustParseBint("1000000000000000000000"),
	mustParseBint("10000000000000000000000"),
	mustParseBint("100000000000000000000000"),
	mustParseBint("1000000000000000000000000"),
	mustParseBint("10000000000000000000000000"),
	mustParseBint("100000000000000000000000000"),
	mustParseBint("1000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("10000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("100000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
	mustParseBint("1000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000"),
}

// bfact is a cache of factorials, where bfact[x] = x!.
var bfact = [...]*bint{
	mustParseBint("1"),
	mustParseBint("1"),
	mustParseBint("2"),
	mustParseBint("6"),
	mustParseBint("24"),
	mustParseBint("120"),
	mustParseBint("720"),
	mustParseBint("5040"),
	mustParseBint("40320"),
	mustParseBint("362880"),
	mustParseBint("3628800"),
	mustParseBint("39916800"),
	mustParseBint("479001600"),
	mustParseBint("6227020800"),
	mustParseBint("87178291200"),
	mustParseBint("1307674368000"),
	mustParseBint("20922789888000"),
	mustParseBint("355687428096000"),
	mustParseBint("6402373705728000"),
	mustParseBint("121645100408832000"),
	mustParseBint("2432902008176640000"),
	mustParseBint("51090942171709440000"),
	mustParseBint("1124000727777607680000"),
	mustParseBint("25852016738884976640000"),
	mustParseBint("620448401733239439360000"),
	mustParseBint("15511210043330985984000000"),
	mustParseBint("403291461126605635584000000"),
	mustParseBint("10888869450418352160768000000"),
	mustParseBint("304888344611713860501504000000"),
	mustParseBint("8841761993739701954543616000000"),
	mustParseBint("265252859812191058636308480000000"),
	mustParseBint("8222838654177922817725562880000000"),
	mustParseBint("263130836933693530167218012160000000"),
	mustParseBint("8683317618811886495518194401280000000"),
	mustParseBint("295232799039604140847618609643520000000"),
	mustParseBint("10333147966386144929666651337523200000000"),
	mustParseBint("371993326789901217467999448150835200000000"),
	mustParseBint("13763753091226345046315979581580902400000000"),
	mustParseBint("523022617466601111760007224100074291200000000"),
	mustParseBint("20397882081197443358640281739902897356800000000"),
	mustParseBint("815915283247897734345611269596115894272000000000"),
	mustParseBint("33452526613163807108170062053440751665152000000000"),
	mustParseBint("1405006117752879898543142606244511569936384000000000"),
	mustParseBint("60415263063373835637355132068513997507264512000000000"),
	mustParseBint("2658271574788448768043625811014615890319638528000000000"),
	mustParseBint("119622220865480194561963161495657715064383733760000000000"),
	mustParseBint("5502622159812088949850305428800254892961651752960000000000"),
	mustParseBint("258623241511168180642964355153611979969197632389120000000000"),
	mustParseBint("12413915592536072670862289047373375038521486354677760000000000"),
	mustParseBint("608281864034267560872252163321295376887552831379210240000000000"),
	mustParseBint("30414093201713378043612608166064768844377641568960512000000000000"),
	mustParseBint("1551118753287382280224243016469303211063259720016986112000000000000"),
	mustParseBint("80658175170943878571660636856403766975289505440883277824000000000000"),
	mustParseBint("4274883284060025564298013753389399649690343788366813724672000000000000"),
	mustParseBint("230843697339241380472092742683027581083278564571807941132288000000000000"),
	mustParseBint("12696403353658275925965100847566516959580321051449436762275840000000000000"),
	mustParseBint("710998587804863451854045647463724949736497978881168458687447040000000000000"),
	mustParseBint("40526919504877216755680601905432322134980384796226602145184481280000000000000"),
	mustParseBint("2350561331282878571829474910515074683828862318181142924420699914240000000000000"),
	mustParseBint("138683118545689835737939019720389406345902876772687432540821294940160000000000000"),
	mustParseBint("8320987112741390144276341183223364380754172606361245952449277696409600000000000000"),
	mustParseBint("507580213877224798800856812176625227226004528988036003099405939480985600000000000000"),
	mustParseBint("31469973260387937525653122354950764088012280797258232192163168247821107200000000000000"),
	mustParseBint("1982608315404440064116146708361898137544773690227268628106279599612729753600000000000000"),
	mustParseBint("126886932185884164103433389335161480802865516174545192198801894375214704230400000000000000"),
	mustParseBint("8247650592082470666723170306785496252186258551345437492922123134388955774976000000000000000"),
	mustParseBint("544344939077443064003729240247842752644293064388798874532860126869671081148416000000000000000"),
	mustParseBint("36471110918188685288249859096605464427167635314049524593701628500267962436943872000000000000000"),
	mustParseBint("2480035542436830599600990418569171581047399201355367672371710738018221445712183296000000000000000"),
	mustParseBint("171122452428141311372468338881272839092270544893520369393648040923257279754140647424000000000000000"),
	mustParseBint("11978571669969891796072783721689098736458938142546425857555362864628009582789845319680000000000000000"),
	mustParseBint("850478588567862317521167644239926010288584608120796235886430763388588680378079017697280000000000000000"),
	mustParseBint("61234458376886086861524070385274672740778091784697328983823014963978384987221689274204160000000000000000"),
	mustParseBint("4470115461512684340891257138125051110076800700282905015819080092370422104067183317016903680000000000000000"),
	mustParseBint("330788544151938641225953028221253782145683251820934971170611926835411235700971565459250872320000000000000000"),
	mustParseBint("24809140811395398091946477116594033660926243886570122837795894512655842677572867409443815424000000000000000000"),
	mustParseBint("1885494701666050254987932260861146558230394535379329335672487982961844043495537923117729972224000000000000000000"),
	mustParseBint("145183092028285869634070784086308284983740379224208358846781574688061991349156420080065207861248000000000000000000"),
	mustParseBint("11324281178206297831457521158732046228731749579488251990048962825668835325234200766245086213177344000000000000000000"),
	mustParseBint("894618213078297528685144171539831652069808216779571907213868063227837990693501860533361810841010176000000000000000000"),
	mustParseBint("71569457046263802294811533723186532165584657342365752577109445058227039255480148842668944867280814080000000000000000000"),
	mustParseBint("5797126020747367985879734231578109105412357244731625958745865049716390179693892056256184534249745940480000000000000000000"),
	mustParseBint("475364333701284174842138206989404946643813294067993328617160934076743994734899148613007131808479167119360000000000000000000"),
	mustParseBint("39455239697206586511897471180120610571436503407643446275224357528369751562996629334879591940103770870906880000000000000000000"),
	mustParseBint("3314240134565353266999387579130131288000666286242049487118846032383059131291716864129885722968716753156177920000000000000000000"),
	mustParseBint("281710411438055027694947944226061159480056634330574206405101912752560026159795933451040286452340924018275123200000000000000000000"),
	mustParseBint("24227095383672732381765523203441259715284870552429381750838764496720162249742450276789464634901319465571660595200000000000000000000"),
	mustParseBint("2107757298379527717213600518699389595229783738061356212322972511214654115727593174080683423236414793504734471782400000000000000000000"),
	mustParseBint("185482642257398439114796845645546284380220968949399346684421580986889562184028199319100141244804501828416633516851200000000000000000000"),
	mustParseBint("16507955160908461081216919262453619309839666236496541854913520707833171034378509739399912570787600662729080382999756800000000000000000000"),
	mustParseBint("1485715964481761497309522733620825737885569961284688766942216863704985393094065876545992131370884059645617234469978112000000000000000000000"),
	mustParseBint("135200152767840296255166568759495142147586866476906677791741734597153670771559994765685283954750449427751168336768008192000000000000000000000"),
	mustParseBint("12438414054641307255475324325873553077577991715875414356840239582938137710983519518443046123837041347353107486982656753664000000000000000000000"),
	mustParseBint("1156772507081641574759205162306240436214753229576413535186142281213246807121467315215203289516844845303838996289387078090752000000000000000000000"),
	mustParseBint("108736615665674308027365285256786601004186803580182872307497374434045199869417927630229109214583415458560865651202385340530688000000000000000000000"),
	mustParseBint("10329978488239059262599702099394727095397746340117372869212250571234293987594703124871765375385424468563282236864226607350415360000000000000000000000"),
	mustParseBint("991677934870949689209571401541893801158183648651267795444376054838492222809091499987689476037000748982075094738965754305639874560000000000000000000000"),
	mustParseBint("96192759682482119853328425949563698712343813919172976158104477319333745612481875498805879175589072651261284189679678167647067832320000000000000000000000"),
	mustParseBint("9426890448883247745626185743057242473809693764078951663494238777294707070023223798882976159207729119823605850588608460429412647567360000000000000000000000"),
}

// bscale is a scale of precomputed values in bexp, blog and bnlog10.
const bscale = 41

// bexp is a cache of powers of e, where bexp[x] = round(exp(x) * 10^41).
var bexp = [...]*bint{
	mustParseBint("100000000000000000000000000000000000000000"),
	mustParseBint("271828182845904523536028747135266249775725"),
	mustParseBint("738905609893065022723042746057500781318032"),
	mustParseBint("2008553692318766774092852965458171789698791"),
	mustParseBint("5459815003314423907811026120286087840279074"),
	mustParseBint("14841315910257660342111558004055227962348767"),
	mustParseBint("40342879349273512260838718054338827960589990"),
	mustParseBint("109663315842845859926372023828812143244221913"),
	mustParseBint("298095798704172827474359209945288867375596794"),
	mustParseBint("810308392757538400770999668943275996501147609"),
	mustParseBint("2202646579480671651695790064528424436635351262"),
	mustParseBint("5987414171519781845532648579225778161426107970"),
	mustParseBint("16275479141900392080800520489848678317020928448"),
	mustParseBint("44241339200892050332610277594908828178439130606"),
	mustParseBint("120260428416477677774923677076785944941248654338"),
	mustParseBint("326901737247211063930185504609172131550573854382"),
	mustParseBint("888611052050787263676302374078145035080271982186"),
	mustParseBint("2415495275357529821477543518038582387986756735272"),
	mustParseBint("6565996913733051113878650325906003356921635578619"),
	mustParseBint("17848230096318726084491003378872270388361973316517"),
	mustParseBint("48516519540979027796910683054154055868463898894485"),
	mustParseBint("131881573448321469720999888374530278509144443738048"),
	mustParseBint("358491284613159156168115994597842068922269306503727"),
	mustParseBint("974480344624890260003463268482297527764938776403601"),
	mustParseBint("2648912212984347229413916215281188234087019861924853"),
	mustParseBint("7200489933738587252416135146612615791522353381339528"),
	mustParseBint("19572960942883876426977639787609534279203610095069759"),
	mustParseBint("53204824060179861668374730434117744165925580428368881"),
	mustParseBint("144625706429147517367704742299692885690206232950992288"),
	mustParseBint("393133429714404207438862058084352768579694233344390219"),
	mustParseBint("1068647458152446214699046865074140165002449500547305499"),
	mustParseBint("2904884966524742523108568211167982566676469509029698025"),
	mustParseBint("7896296018268069516097802263510822421995619511535233066"),
	mustParseBint("21464357978591606462429776153126088036922590605479789726"),
	mustParseBint("58346174252745488140290273461039101900365923894110810578"),
	mustParseBint("158601345231343072812964462577466012517620395013452615427"),
	mustParseBint("431123154711519522711342229285692539078886361678034773077"),
	mustParseBint("1171914237280261130877293979119019452167536369446182238348"),
	mustParseBint("3185593175711375622032867170129864599954220990518100775324"),
	mustParseBint("8659340042399374695360693271926493424970185470019598659152"),
	mustParseBint("23538526683701998540789991074903480450887161725455546723665"),
	mustParseBint("63984349353005494922266340351557081887933662139685527945496"),
	mustParseBint("173927494152050104739468130361123522614798405772500840103706"),
	mustParseBint("472783946822934656147445756274428037081975196238093817096720"),
	mustParseBint("1285160011435930827580929963214309925780114322075882587192003"),
	mustParseBint("3493427105748509534803479723340609953341165649751815426012609"),
	mustParseBint("9496119420602448874513364911711832310181715892107998785043817"),
	mustParseBint("25813128861900673962328580021527338043163708299304406081061397"),
	mustParseBint("70167359120976317386547159988611740545593799872532198375454888"),
	mustParseBint("190734657249509969052509984095384844738818973054378340247523471"),
	mustParseBint("518470552858707246408745332293348538482746910058384640190405693"),
	mustParseBint("1409349082426938796449214331237016878868476079006371401705261930"),
	mustParseBint("3831008000716576849303569548786199389870559745511657759083460282"),
	mustParseBint("10413759433029087797183472933493796439804671331549132480520877196"),
	mustParseBint("28307533032746939004420635480140745408503258935359068407207313389"),
	mustParseBint("76947852651420171381827455901293939920707675726508409192316811729"),
	mustParseBint("209165949601299615390707115721467377815296684749926726566766760571"),
	mustParseBint("568571999933593222264034882063325330337215833448821774348912207348"),
	mustParseBint("1545538935590103930353076691117462006836371709693083024671167680619"),
	mustParseBint("4201210403790514254956593430719161768411111973299096786464257294200"),
	mustParseBint("11420073898156842836629571831447656301980459595563958395650279917582"),
	mustParseBint("31042979357019199087073421411071003720629501767637088151075640107741"),
	mustParseBint("84383566687414544890733294803731179600806926796674104046181781869882"),
	mustParseBint("229378315946960987909935284026861360046327934090304823013811995111671"),
	mustParseBint("623514908081161688290923870892846974483139184623579991438859169901398"),
	mustParseBint("1694889244410333714141783611437197494892623622551650491315726964531624"),
	mustParseBint("4607186634331291542677318442806008689334900303709604073064405455741671"),
	mustParseBint("12523631708422137805135219607443657675348852740449748164080228572102416"),
	mustParseBint("34042760499317405213769071870043505953738761399456516360515457366001246"),
	mustParseBint("92537817255877876002423979166873458734765992072084131193538452395984132"),
	mustParseBint("251543867091916700626578117425211296140741297338314051382184015698611046"),
	mustParseBint("683767122976274386675589282667771095594583593128908772566867276581148527"),
	mustParseBint("1858671745284127980340370181254541194694639669623280326839338508664990931"),
	mustParseBint("5052393630276104194557038332185764648536724497561717707892131665519301059"),
	mustParseBint("13733829795401761877841885298085389315799830792499374638225617048580357052"),
	mustParseBint("37332419967990016402549083172647001434277836869160000233938286724489039463"),
	mustParseBint("101480038811388872783246178413171697577666481516811788087636356532265354778"),
	mustParseBint("275851345452317020628646981990266194334152017524870885909958836251105051085"),
	mustParseBint("749841699699012043467563059122406045470465897778615881301712227912805589523"),
	mustParseBint("2038281066512668766832313753717263237469736486152421023534282361781159646993"),
	mustParseBint("5540622384393510052571173395831661292485672883268532287030018828204570044975"),
	mustParseBint("15060973145850305483525941301676749818993964921282654216298674950430270913981"),
	mustParseBint("40939969621274546966609142293278290432005379219385277776474289917001253733742"),
	mustParseBint("111286375479175941208707147818394080573407692072049303419300322944399234801420"),
	mustParseBint("302507732220114233826656639644342874246903222230408816021160842116018977688417"),
	mustParseBint("822301271462291351030432801640777469548628618847627403952393384974416103487573"),
	mustParseBint("2235246603734715047443065732332714739877539251320997394294418376804179728019707"),
	mustParseBint("6076030225056872149522328938130276075261375325962121014840857614538080439681426"),
	mustParseBint("16516362549940018555283297962648587670696288420000448138888115075308155590848548"),
	mustParseBint("44896128191743452462842455796453162777659805863718515260493172176526871508945617"),
	mustParseBint("122040329431784080200271003513636975397074642109976754624434382982431272735857720"),
	mustParseBint("331740009833574262575551610785259190960301458118233082704880726006914701279896785"),
	mustParseBint("901762840503429893140099598217090525912874611320072683317921490664655375792186090"),
	mustParseBint("2451245542920085785552772943110915342348756414964690609545833883604150632588240812"),
	mustParseBint("6663176216410895834244814050240873262687388754209656559399850257165488523386744387"),
	mustParseBint("18112390828890232821937987580988159250478971745441286187030605794738143642445974469"),
	mustParseBint("49234582860120583997548620591133044948377995712526602794095096218650199367793067780"),
	mustParseBint("133833471920426950046173640870611502907671745721364460012055487319044738767664950934"),
	mustParseBint("363797094760880457928774382676018572989310125586408296287549241498281217782703355658"),
	mustParseBint("988903031934694677056003096713803710140508260719933517340199715371109444700740600601"),
}

// blog is a cache of natural logarithms, where blog[x] = round(ln(x) * 10^41).
var blog = [...]*bint{
	nil,
	mustParseBint("0"),
	mustParseBint("69314718055994530941723212145817656807550"),
	mustParseBint("109861228866810969139524523692252570464749"),
	mustParseBint("138629436111989061883446424291635313615100"),
	mustParseBint("160943791243410037460075933322618763952560"),
	mustParseBint("179175946922805500081247735838070227272299"),
	mustParseBint("194591014905531330510535274344317972963708"),
	mustParseBint("207944154167983592825169636437452970422650"),
	mustParseBint("219722457733621938279049047384505140929498"),
	mustParseBint("230258509299404568401799145468436420760110"),
	mustParseBint("239789527279837054406194357796512929982171"),
	mustParseBint("248490664978800031022970947983887884079849"),
	mustParseBint("256494935746153673605348744156531860480527"),
	mustParseBint("263905732961525861452258486490135629771258"),
	mustParseBint("270805020110221006599600457014871334417309"),
	mustParseBint("277258872223978123766892848583270627230200"),
	mustParseBint("283321334405621608024953461787312653558820"),
	mustParseBint("289037175789616469220772259530322797737048"),
	mustParseBint("294443897916644046000902743188785353723738"),
	mustParseBint("299573227355399099343522357614254077567660"),
	mustParseBint("304452243772342299650059798036570543428458"),
	mustParseBint("309104245335831585347917569942330586789721"),
	mustParseBint("313549421592914969080675283181019611844238"),
	mustParseBint("317805383034794561964694160129705540887399"),
	mustParseBint("321887582486820074920151866645237527905120"),
	mustParseBint("325809653802148204547071956302349517288077"),
	mustParseBint("329583686600432907418573571076757711394247"),
	mustParseBint("333220451017520392393981698635953286578808"),
	mustParseBint("336729582998647402718327203236191160549451"),
	mustParseBint("340119738166215537541323669160688991224859"),
	mustParseBint("343398720448514624592916432454235721044994"),
	mustParseBint("346573590279972654708616060729088284037750"),
	mustParseBint("349650756146648023545718881488765500446920"),
	mustParseBint("352636052461616138966676673933130310366370"),
	mustParseBint("355534806148941367970611207666936736916269"),
	mustParseBint("358351893845611000162495471676140454544598"),
	mustParseBint("361091791264422444436809567103144716390008"),
	mustParseBint("363758615972638576942625955334603010531288"),
	mustParseBint("366356164612964642744873267848784430945276"),
	mustParseBint("368887945411393630285245569760071734375210"),
	mustParseBint("371357206670430780386676337303740758837641"),
	mustParseBint("373766961828336830591783010182388200236008"),
	mustParseBint("376120011569356242347284251334584703555914"),
	mustParseBint("378418963391826116289640782088148243597271"),
	mustParseBint("380666248977031975739124980707123904882058"),
	mustParseBint("382864139648909500022398495326837268651788"),
	mustParseBint("385014760171005858682095066977217370889605"),
	mustParseBint("387120101090789092906417372275523197694949"),
	mustParseBint("389182029811062661021070548688635945927417"),
	mustParseBint("391202300542814605861875078791055184712670"),
	mustParseBint("393182563272432577164477985479565224023569"),
	mustParseBint("395124371858142735488795168448167174095627"),
	mustParseBint("397029191355212183414446913902905777035998"),
	mustParseBint("398898404656427438360296783222575368201797"),
	mustParseBint("400733318523247091866270291119131693934731"),
	mustParseBint("402535169073514923335704910781770943386359"),
	mustParseBint("404305126783455015140427266881037924188487"),
	mustParseBint("406044301054641933660050415382008817357001"),
	mustParseBint("407753744390571945061605037371969762406335"),
	mustParseBint("409434456222210068483046881306506648032409"),
	mustParseBint("411087386417331124875138910342561474631568"),
	mustParseBint("412713438504509155534639644600053377852544"),
	mustParseBint("414313472639153268789584321728823113893207"),
	mustParseBint("415888308335967185650339272874905940845300"),
	mustParseBint("417438726989563711065424677479150624433087"),
	mustParseBint("418965474202642554487442093634583157254470"),
	mustParseBint("420469261939096605967007199636372275056693"),
	mustParseBint("421950770517610669908399886078947967173920"),
	mustParseBint("423410650459725938220199806873272182308987"),
	mustParseBint("424849524204935898912334419812754393723819"),
	mustParseBint("426267987704131542132945453251303409675958"),
	mustParseBint("427666611901605531104218683821958111352148"),
	mustParseBint("429045944114839112909210885743854257090475"),
	mustParseBint("430406509320416975378532779248962373197558"),
	mustParseBint("431748811353631044059676390337490098369869"),
	mustParseBint("433073334028633107884349167480420667338838"),
	mustParseBint("434380542185368384916729632140830902945879"),
	mustParseBint("435670882668959173686596479994602087752826"),
	mustParseBint("436944785246702149417294554148141092217354"),
	mustParseBint("438202663467388161226968781905889391182760"),
	mustParseBint("439444915467243876558098094769010281858996"),
	mustParseBint("440671924726425311328399549449558415645191"),
	mustParseBint("441884060779659792347547222329137045302931"),
	mustParseBint("443081679884331361533506222328205857043558"),
	mustParseBint("444265125649031645485029395109931417511380"),
	mustParseBint("445434729625350773289007463480402360363464"),
	mustParseBint("446590811865458371857851726928443731014200"),
	mustParseBint("447733681447820647231363994233965900404821"),
	mustParseBint("448863636973213983831781554066984921940466"),
	mustParseBint("449980967033026506680848192852941561689608"),
	mustParseBint("451085950651685004115884018500849833444235"),
	mustParseBint("452178857704904030964121707472654925459338"),
	mustParseBint("453259949315325593732440956146488291509743"),
	mustParseBint("454329478227000389623818279123035027697155"),
	mustParseBint("455387689160054083460978676511404117676298"),
	mustParseBint("456434819146783623848140584421340854502499"),
	mustParseBint("457471097850338282211672162170396171380891"),
	mustParseBint("458496747867057191962793760834453602734967"),
	mustParseBint("459511985013458992685243405181018070911669"),
}

// bnlog10 is a cache of  multiples of the natural logarithm of 10, where bnlog10[x] = round(x * ln(10) * 10^41).
var bnlog10 = [...]*bint{
	mustParseBint("0"),
	mustParseBint("230258509299404568401799145468436420760110"),
	mustParseBint("460517018598809136803598290936872841520220"),
	mustParseBint("690775527898213705205397436405309262280330"),
	mustParseBint("921034037197618273607196581873745683040441"),
	mustParseBint("1151292546497022842008995727342182103800551"),
	mustParseBint("1381551055796427410410794872810618524560661"),
	mustParseBint("1611809565095831978812594018279054945320771"),
	mustParseBint("1842068074395236547214393163747491366080881"),
	mustParseBint("2072326583694641115616192309215927786840991"),
	mustParseBint("2302585092994045684017991454684364207601101"),
	mustParseBint("2532843602293450252419790600152800628361212"),
	mustParseBint("2763102111592854820821589745621237049121322"),
	mustParseBint("2993360620892259389223388891089673469881432"),
	mustParseBint("3223619130191663957625188036558109890641542"),
	mustParseBint("3453877639491068526026987182026546311401652"),
	mustParseBint("3684136148790473094428786327494982732161762"),
	mustParseBint("3914394658089877662830585472963419152921873"),
	mustParseBint("4144653167389282231232384618431855573681983"),
	mustParseBint("4374911676688686799634183763900291994442093"),
	mustParseBint("4605170185988091368035982909368728415202203"),
	mustParseBint("4835428695287495936437782054837164835962313"),
	mustParseBint("5065687204586900504839581200305601256722423"),
	mustParseBint("5295945713886305073241380345774037677482533"),
	mustParseBint("5526204223185709641643179491242474098242644"),
	mustParseBint("5756462732485114210044978636710910519002754"),
	mustParseBint("5986721241784518778446777782179346939762864"),
	mustParseBint("6216979751083923346848576927647783360522974"),
	mustParseBint("6447238260383327915250376073116219781283084"),
	mustParseBint("6677496769682732483652175218584656202043194"),
	mustParseBint("6907755278982137052053974364053092622803304"),
	mustParseBint("7138013788281541620455773509521529043563415"),
	mustParseBint("7368272297580946188857572654989965464323525"),
	mustParseBint("7598530806880350757259371800458401885083635"),
	mustParseBint("7828789316179755325661170945926838305843745"),
	mustParseBint("8059047825479159894062970091395274726603855"),
	mustParseBint("8289306334778564462464769236863711147363965"),
	mustParseBint("8519564844077969030866568382332147568124076"),
	mustParseBint("8749823353377373599268367527800583988884186"),
	mustParseBint("8980081862676778167670166673269020409644296"),
	mustParseBint("9210340371976182736071965818737456830404406"),
	mustParseBint("9440598881275587304473764964205893251164516"),
	mustParseBint("9670857390574991872875564109674329671924626"),
	mustParseBint("9901115899874396441277363255142766092684736"),
	mustParseBint("10131374409173801009679162400611202513444847"),
	mustParseBint("10361632918473205578080961546079638934204957"),
	mustParseBint("10591891427772610146482760691548075354965067"),
	mustParseBint("10822149937072014714884559837016511775725177"),
	mustParseBint("11052408446371419283286358982484948196485287"),
	mustParseBint("11282666955670823851688158127953384617245397"),
	mustParseBint("11512925464970228420089957273421821038005507"),
	mustParseBint("11743183974269632988491756418890257458765618"),
	mustParseBint("11973442483569037556893555564358693879525728"),
	mustParseBint("12203700992868442125295354709827130300285838"),
	mustParseBint("12433959502167846693697153855295566721045948"),
	mustParseBint("12664218011467251262098953000764003141806058"),
	mustParseBint("12894476520766655830500752146232439562566168"),
	mustParseBint("13124735030066060398902551291700875983326278"),
	mustParseBint("13354993539365464967304350437169312404086389"),
	mustParseBint("13585252048664869535706149582637748824846499"),
	mustParseBint("13815510557964274104107948728106185245606609"),
	mustParseBint("14045769067263678672509747873574621666366719"),
	mustParseBint("14276027576563083240911547019043058087126829"),
	mustParseBint("14506286085862487809313346164511494507886939"),
	mustParseBint("14736544595161892377715145309979930928647050"),
	mustParseBint("14966803104461296946116944455448367349407160"),
	mustParseBint("15197061613760701514518743600916803770167270"),
	mustParseBint("15427320123060106082920542746385240190927380"),
	mustParseBint("15657578632359510651322341891853676611687490"),
	mustParseBint("15887837141658915219724141037322113032447600"),
	mustParseBint("16118095650958319788125940182790549453207710"),
	mustParseBint("16348354160257724356527739328258985873967821"),
	mustParseBint("16578612669557128924929538473727422294727931"),
	mustParseBint("16808871178856533493331337619195858715488041"),
	mustParseBint("17039129688155938061733136764664295136248151"),
	mustParseBint("17269388197455342630134935910132731557008261"),
	mustParseBint("17499646706754747198536735055601167977768371"),
	mustParseBint("17729905216054151766938534201069604398528481"),
	mustParseBint("17960163725353556335340333346538040819288592"),
	mustParseBint("18190422234652960903742132492006477240048702"),
	mustParseBint("18420680743952365472143931637474913660808812"),
	mustParseBint("18650939253251770040545730782943350081568922"),
	mustParseBint("18881197762551174608947529928411786502329032"),
	mustParseBint("19111456271850579177349329073880222923089142"),
	mustParseBint("19341714781149983745751128219348659343849253"),
	mustParseBint("19571973290449388314152927364817095764609363"),
	mustParseBint("19802231799748792882554726510285532185369473"),
	mustParseBint("20032490309048197450956525655753968606129583"),
	mustParseBint("20262748818347602019358324801222405026889693"),
	mustParseBint("20493007327647006587760123946690841447649803"),
	mustParseBint("20723265836946411156161923092159277868409913"),
	mustParseBint("20953524346245815724563722237627714289170024"),
	mustParseBint("21183782855545220292965521383096150709930134"),
	mustParseBint("21414041364844624861367320528564587130690244"),
	mustParseBint("21644299874144029429769119674033023551450354"),
	mustParseBint("21874558383443433998170918819501459972210464"),
	mustParseBint("22104816892742838566572717964969896392970574"),
	mustParseBint("22335075402042243134974517110438332813730684"),
	mustParseBint("22565333911341647703376316255906769234490795"),
	mustParseBint("22795592420641052271778115401375205655250905"),
}

// mustParseBint converts a string to *big.Int, panicking on error.
// Use only for package variable initialization and test code!
func mustParseBint(s string) *bint {
	z, ok := new(big.Int).SetString(s, 10)
	if !ok {
		panic(fmt.Errorf("mustParseBint(%q) failed: parsing error", s))
	}
	if z.Sign() < 0 {
		panic(fmt.Errorf("mustParseBint(%q) failed: negative number", s))
	}
	return (*bint)(z)
}

func (z *bint) sign() int {
	return (*big.Int)(z).Sign()
}

func (z *bint) cmp(x *bint) int {
	return (*big.Int)(z).Cmp((*big.Int)(x))
}

func (z *bint) string() string {
	return (*big.Int)(z).String()
}

func (z *bint) setBint(x *bint) {
	(*big.Int)(z).Set((*big.Int)(x))
}

func (z *bint) setInt64(x int64) {
	(*big.Int)(z).SetInt64(x)
}

func (z *bint) setBytes(buf []byte) {
	(*big.Int)(z).SetBytes(buf)
}

func (z *bint) setFint(x fint) {
	(*big.Int)(z).SetUint64(uint64(x))
}

// fint converts *big.Int to uint64.
// If z cannot be represented as uint64, the result is undefined.
func (z *bint) fint() fint {
	f := (*big.Int)(z).Uint64()
	return fint(f)
}

// add calculates z = x + y.
func (z *bint) add(x, y *bint) {
	(*big.Int)(z).Add((*big.Int)(x), (*big.Int)(y))
}

// inc calcualtes z = x + 1.
func (z *bint) inc(x *bint) {
	y := bpow10[0]
	z.add(x, y)
}

// sub calculates z = x - y.
func (z *bint) sub(x, y *bint) {
	(*big.Int)(z).Sub((*big.Int)(x), (*big.Int)(y))
}

// subAbs calculates z = |x - y|.
func (z *bint) subAbs(x, y *bint) {
	switch x.cmp(y) {
	case 1:
		z.sub(x, y)
	default:
		z.sub(y, x)
	}
}

// dbl (Double) calculates z = x * 2.
func (z *bint) dbl(x *bint) {
	(*big.Int)(z).Lsh((*big.Int)(x), 1)
}

// hlf (Half) calculates z = ⌊x / 2⌋.
func (z *bint) hlf(x *bint) {
	(*big.Int)(z).Rsh((*big.Int)(x), 1)
}

// mul calculates z = x * y.
func (z *bint) mul(x, y *bint) {
	// Copying x, y to prevent heap allocations.
	if z == x {
		b := getBint()
		defer putBint(b)
		b.setBint(x)
		x = b
	}
	if z == y {
		b := getBint()
		defer putBint(b)
		b.setBint(y)
		y = b
	}
	(*big.Int)(z).Mul((*big.Int)(x), (*big.Int)(y))
}

// pow calculates z = x^y.
// If y is negative, the result is unpredictable.
func (z *bint) pow(x, y *bint) {
	(*big.Int)(z).Exp((*big.Int)(x), (*big.Int)(y), nil)
}

// pow10 calculates z = 10^power.
// If power is negative, the result is unpredictable.
func (z *bint) pow10(power int) {
	x := getBint()
	defer putBint(x)
	x.setInt64(10)
	y := getBint()
	defer putBint(y)
	y.setInt64(int64(power))
	z.pow(x, y)
}

// quo calculates z = ⌊x / y⌋.
func (z *bint) quo(x, y *bint) {
	// Passing r to prevent heap allocations.
	r := getBint()
	defer putBint(r)
	z.quoRem(x, y, r)
}

// quoRem calculates z = ⌊x / y⌋, r = x - y * z.
func (z *bint) quoRem(x, y, r *bint) {
	(*big.Int)(z).QuoRem((*big.Int)(x), (*big.Int)(y), (*big.Int)(r))
}

func (z *bint) isOdd() bool {
	return (*big.Int)(z).Bit(0) != 0
}

// lsh (Left Shift) calculates z = x * 10^shift.
func (z *bint) lsh(x *bint, shift int) {
	var y *bint
	if shift < len(bpow10) {
		y = bpow10[shift]
	} else {
		y = getBint()
		defer putBint(y)
		y.pow10(shift)
	}
	z.mul(x, y)
}

// fsa (Fused Shift and Addition) calculates z = x * 10^shift + f.
func (z *bint) fsa(x *bint, shift int, f fint) {
	y := getBint()
	defer putBint(y)
	y.setFint(f)
	z.lsh(x, shift)
	z.add(z, y)
}

// rshDown (Right Shift) calculates z = ⌊x / 10^shift⌋ and rounds
// result towards zero.
func (z *bint) rshDown(x *bint, shift int) {
	// Special cases
	switch {
	case x.sign() == 0:
		z.setFint(0)
		return
	case shift <= 0:
		z.setBint(x)
		return
	}
	// General case
	var y *bint
	if shift < len(bpow10) {
		y = bpow10[shift]
	} else {
		y = getBint()
		defer putBint(y)
		y.pow10(shift)
	}
	z.quo(x, y)
}

// rshHalfEven (Right Shift) calculates z = round(x / 10^shift) and
// rounds result using "half to even" rule.
func (z *bint) rshHalfEven(x *bint, shift int) {
	// Special cases
	switch {
	case x.sign() == 0:
		z.setFint(0)
		return
	case shift <= 0:
		z.setBint(x)
		return
	}
	// General case
	var y, r *bint
	r = getBint()
	defer putBint(r)
	if shift < len(bpow10) {
		y = bpow10[shift]
	} else {
		y = getBint()
		defer putBint(y)
		y.pow10(shift)
	}
	z.quoRem(x, y, r)
	r.dbl(r) // r = r * 2
	switch y.cmp(r) {
	case -1:
		z.inc(z) // z = z + 1
	case 0:
		// half-to-even
		if z.isOdd() {
			z.inc(z) // z = z + 1
		}
	}
}

// prec returns length of z in decimal digits.
// prec assumes that 0 has no digits.
// If z is negative, the result is unpredictable.
//
// z.prec() is significantly faster than len(z.string()),
// if z has less than len(bpow10) digits.
func (z *bint) prec() int {
	// Special case
	if z.cmp(bpow10[len(bpow10)-1]) > 0 {
		return len(z.string())
	}
	// General case
	left, right := 0, len(bpow10)
	for left < right {
		mid := (left + right) / 2
		if z.cmp(bpow10[mid]) < 0 {
			right = mid
		} else {
			left = mid + 1
		}
	}
	return left
}

// hasPrec checks if z has a given number of digits or more.
// hasPrec assumes that 0 has no digits.
// If z is negative, the result is unpredictable.
//
// z.hasPrec(p) is significantly faster than z.prec() >= p,
// if z has no more than len(bpow10) digits.
func (z *bint) hasPrec(prec int) bool {
	// Special cases
	switch {
	case prec < 1:
		return true
	case prec > len(bpow10):
		return len(z.string()) >= prec
	}
	// General case
	return z.cmp(bpow10[prec-1]) >= 0
}

// bpool is a cache of reusable *big.Int instances.
var bpool = sync.Pool{
	New: func() any {
		return (*bint)(new(big.Int))
	},
}

// getBint obtains a *big.Int from the pool.
func getBint() *bint {
	return bpool.Get().(*bint)
}

// putBint returns the *big.Int into the pool.
func putBint(b *bint) {
	bpool.Put(b)
}
