/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   THIRD PARTY LICENSE NOTICE:

   CancelToken is derived from the implementation of CancellationToken in
   Promise Extensions for Javascript: https://github.com/rbuckton/prex

   Promise Extensions is licensed under the Apache 2.0 License:

   Promise Extensions for JavaScript
   Copyright (c) Microsoft Corporation

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
import { Cancelable, CancelableSource, CancelSignal, CancelSubscription } from "@esfx/cancelable";
import { Disposable } from "@esfx/disposable";
export { CancelError, CancelSubscription } from "@esfx/cancelable";
declare type DOMAbortSignal = (typeof globalThis) extends {
    AbortSignal: Function & {
        prototype: infer TAbortSignal;
    };
} ? TAbortSignal : never;
declare const kEventTarget: unique symbol;
/**
 * Signals a {@link CancelToken} when cancellation has been requested.
 */
export interface CancelSource extends CancelableSource, Disposable {
    /**
     * Gets the {@link CancelToken} linked to this source.
     */
    readonly token: CancelToken;
    /**
     * Cancels the source, evaluating any subscribed callbacks. If any callback raises an exception,
     * the exception is propagated to a host specific unhandled exception mechanism.
     */
    cancel(reason?: unknown): void;
    /**
     * Closes the source, preventing the possibility of future cancellation.
     */
    close(): void;
}
/**
 * Propagates notifications that operations should be canceled.
 */
export declare class CancelToken implements Cancelable, CancelSignal {
    static [kEventTarget]: boolean;
    static readonly none: CancelToken;
    static readonly canceled: CancelToken;
    private constructor();
    /**
     * Gets a value indicating whether the token is signaled.
     */
    get signaled(): boolean;
    /**
     * Gets the reason for cancellation.
     */
    get reason(): unknown;
    /**
     * Gets a value indicating whether the token can be signaled.
     */
    get canBeSignaled(): boolean;
    /**
     * Adapts a {@link CancelToken} from a cancelable.
     */
    static from(cancelable: Cancelable | DOMAbortSignal | null | undefined): CancelToken;
    /**
     * Creates a new {@link CancelSource}.
     *
     * @param cancelables An optional iterable of {@link @esfx/cancelable!Cancelable} objects. If present,
     * the source becomes linked to the provided cancelables and will be canceled
     * when a linked cancelable is canceled.
     *
     * @remarks Calling {@link source} with `cancelables` is similar to {@link race}, except you can
     * individually cancel or close the resulting source. This can be better for memory or GC purposes,
     * since when the resulting source is canceled or closed it can be unlinked from the cancelables,
     * removing references from each cancelable to the resulting source which could otherwise prevent
     * garbage collection.
     */
    static source(cancelables?: Iterable<Cancelable | DOMAbortSignal | null | undefined>): CancelSource;
    /**
     * Returns a {@link CancelToken} that becomes signaled when any of the provided cancelables are signaled.
     *
     * @param cancelables An iterable of {@link @esfx/cancelable!Cancelable} objects.
     *
     * @remarks This is similar to calling {@link source} with the provided `cancelables`. In general,
     * calling {@link source} is preferred as it provides better resource management.
     */
    static race(cancelables: Iterable<Cancelable | DOMAbortSignal | null | undefined>): CancelToken;
    /**
     * Returns a {@link CancelToken} that becomes signaled when all of the provided cancelables are signaled.
     *
     * @param cancelables An iterable of {@link @esfx/cancelable!Cancelable} objects.
     */
    static all(cancelables: Iterable<Cancelable | DOMAbortSignal | null | undefined>): CancelToken;
    /**
     * Gets a {@link CancelToken} that is already canceled with the provided reason.
     */
    static canceledWith(reason: unknown): CancelToken;
    /**
     * Gets a {@link CancelToken} that will be canceled with the provided reason after a timeout has elapsed.
     */
    static timeout(ms: number, reason?: unknown): CancelToken;
    /**
     * Throws a {@link @esfx/cancelable!CancelError} if the token was signaled.
     */
    throwIfSignaled(): void;
    /**
     * Subscribes to notifications for when the object becomes signaled.
     */
    subscribe(onSignaled: () => void): CancelSubscription;
    /** {@inheritDoc @esfx/cancelable!Cancelable#[Cancelable.cancelSignal]:member(1)} */
    [Cancelable.cancelSignal](): CancelToken;
}
//# sourceMappingURL=index.d.ts.map