"use strict";
/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.Disposable = void 0;
const utils_js_1 = require("./internal/utils.js");
/**
 * Indicates an object that has resources that can be explicitly disposed.
 */
class Disposable {
    /**
     * Creates a `Disposable` wrapper around a callback used to dispose of a resource.
     * @deprecated Use `DisposableStack` or `{ [Disposable.dispose]() { ... } }` instead.
     */
    constructor(dispose) {
        if (!(typeof dispose === "function"))
            throw new TypeError("Function expected: dispose");
        return Disposable.create(dispose);
    }
}
exports.Disposable = Disposable;
/**
 * Indicates an object that has resources that can be explicitly disposed.
 */
(function (Disposable) {
    /**
     * A well-known symbol used to define an explicit resource disposal method on an object.
     *
     * NOTE: Uses `Symbol.dispose` if present.
     */
    Disposable.dispose = typeof Symbol["dispose"] === "symbol" ?
        Symbol["dispose"] :
        Symbol.for("@esfx/disposable:Disposable.dispose");
    /**
     * Emulate `using const` using `for..of`.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     *
     * @example
     * ```ts
     * // with `using const` (proposed)
     * {
     *   ...
     *   using const x = expr, y = expr;
     *   ...
     * }
     *
     * // with `Disposable.scope()`:
     * for (const { using, fail } of Disposable.scope()) {
     *   try {
     *     ...
     *     const x = using(expr), y = using(expr);
     *     ...
     *   }
     *   catch (e) {
     *     fail(e);
     *   }
     * }
     * ```
     */
    function* scope() {
        const context = (0, utils_js_1.CreateScope)("sync");
        try {
            context.state = "initialized";
            yield context.scope;
            context.state = "exiting";
        }
        finally {
            context.state = "done";
            (0, utils_js_1.DisposeResources)("sync", context.disposables, context.throwCompletion);
        }
    }
    Disposable.scope = scope;
    /**
     * Yields each disposable in the iterable, disposing it when the generator resumes.
     *
     * This emulates `for (using const x of expr)`.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     *
     * @example
     * ```ts
     * // with `using const` (proposed)
     * for (using const x of expr) {
     *   ...
     * }
     *
     * // with `Disposable.usingEach()`:
     * for (const x of Disposable.usingEach(expr)) {
     *   ...
     * }
     * ```
     */
    function* usingEach(iterable) {
        if (!((typeof iterable === "object" && iterable !== null || typeof iterable === "function") && Symbol.iterator in iterable))
            throw new TypeError("Object not iterable: iterable");
        // for (using const disposable of disposables) yield disposable;
        for (const disposable of iterable) {
            for (const { using, fail } of Disposable.scope())
                try {
                    yield using(disposable);
                }
                catch (e) {
                    fail(e);
                }
        }
    }
    Disposable.usingEach = usingEach;
    const disposablePrototype = Disposable.prototype;
    Object.defineProperty(disposablePrototype, Symbol.toStringTag, { configurable: true, value: "Disposable" });
    /**
     * Creates a `Disposable` wrapper around a callback used to dispose of a resource.
     *
     * NOTE: This is not spec-compliant and will not be standardized. It is preferred to use a `DisposableStack`
     * or to implement `Disposable.dispose` yourself instead.
     */
    function create(dispose) {
        if (!(typeof dispose === "function"))
            throw new TypeError("Function expected: dispose");
        let disposed = false;
        return Object.setPrototypeOf({
            [Disposable.dispose]() {
                if (!disposed) {
                    disposed = true;
                    const cb = dispose;
                    dispose = undefined;
                    cb();
                }
            }
        }, disposablePrototype);
    }
    Disposable.create = create;
    /**
     * Determines whether a value is Disposable.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     */
    function hasInstance(value) {
        return (typeof value === "object" && value !== null || typeof value === "function") && Disposable.dispose in value;
    }
    Disposable.hasInstance = hasInstance;
})(Disposable = exports.Disposable || (exports.Disposable = {}));
Object.defineProperty(Disposable, Symbol.hasInstance, Object.getOwnPropertyDescriptor(Disposable, "hasInstance"));
//# sourceMappingURL=disposable.js.map