/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
import { AsyncDisposable } from "./asyncDisposable.js";
import { Disposable } from "./disposable.js";
export declare type AsyncDisposableLike = AsyncDisposable | Disposable | (() => void | PromiseLike<void>);
/**
 * Emulates Python's `AsyncExitStack`
 */
export declare class AsyncDisposableStack implements AsyncDisposable {
    [Symbol.toStringTag]: string;
    constructor();
    static get [Symbol.species](): typeof AsyncDisposableStack;
    /**
     * Dispose this object's resources.
     *
     * NOTE: `disposeAsync` returns a bound method, so it can be extracted from `AsyncDisposableStack` and called independently:
     *
     * ```ts
     * const stack = new AsyncDisposableStack();
     * for (const f of files) stack.use(openFile(f));
     * const closeFiles = stack.disposeAsync;
     * ...
     * closeFiles();
     * ```
     */
    get disposeAsync(): () => Promise<void>;
    /**
     * Pushes a new disposable resource onto the disposable stack stack. Resources are disposed in the reverse order they were entered.
     * @param value The resource to add.
     * @returns The resource provided.
     */
    use<T extends AsyncDisposableLike | null | undefined>(value: T): T;
    /**
     * Pushes a new disposable resource onto the disposable stack stack. Resources are disposed in the reverse order they were entered.
     * @param value The resource to add.
     * @param onDispose The operation to perform when the resource is disposed.
     * @returns The resource provided.
     */
    use<T>(value: T, onDisposeAsync: (value: T) => void | PromiseLike<void>): T;
    /**
     * Moves all resources out of this `AsyncDisposableStack` and into a new `AsyncDisposableStack` and returns it.
     */
    move(): AsyncDisposableStack;
    /**
     * Dispose this object's resources.
     */
    [AsyncDisposable.asyncDispose](): Promise<void>;
}
//# sourceMappingURL=asyncDisposableStack.d.ts.map