/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
import { Disposable } from "@esfx/disposable";
declare type ErrorOptions = 2 extends ConstructorParameters<typeof Error>["length"] ? ConstructorParameters<typeof Error> extends [message?: string, options?: infer O] ? NonNullable<O> : never : never;
export declare class CancelError extends Error {
    constructor(message?: string, options?: ErrorOptions);
}
/**
 * An object used to unsubscribe from a cancellation signal
 */
export interface CancelSubscription extends Disposable {
    /**
     * Unsubscribes from a cancellation signal.
     */
    unsubscribe(): void;
}
export declare namespace CancelSubscription {
    /**
     * Creates a `CancelSubscription` object for an `unsubscribe` callback.
     * @param unsubscribe The callback to execute when the `unsubscribe()` method is called.
     */
    function create(unsubscribe: () => void): CancelSubscription;
}
/**
 * An object that can be canceled from an external source.
 */
export interface Cancelable {
    /**
     * Gets the CancelSignal for this Cancelable.
     */
    [Cancelable.cancelSignal](): CancelSignal;
}
export declare namespace Cancelable {
    /**
     * A well-known symbol used to define a method to retrieve the `CancelSignal` for an object.
     */
    const cancelSignal: unique symbol;
    /**
     * A `Cancelable` that is already signaled.
     */
    const canceled: CancelableCancelSignal;
    /**
     * A `Cancelable` that can never be signaled.
     */
    const none: CancelableCancelSignal;
    /**
     * Determines whether `cancelable` is in the signaled state.
     */
    function isSignaled(cancelable: Cancelable | null | undefined): boolean;
    /**
     * Gets the reason for cancelation.
     */
    function getReason(cancelable: Cancelable | null | undefined): unknown;
    /**
     * Throws a `CancelError` exception if the provided `cancelable` is in the signaled state.
     */
    function throwIfSignaled(cancelable: Cancelable | null | undefined): void;
    /**
     * Subscribes to be notified when a `cancelable` becomes signaled.
     */
    function subscribe(cancelable: Cancelable | null | undefined, onSignaled: () => void): CancelSubscription;
    const name = "Cancelable";
    /**
     * Determines whether a value is a `Cancelable` object.
     */
    function hasInstance(value: unknown): value is Cancelable;
}
/**
 * An object that represents a cancellation signal.
 */
export interface CancelSignal {
    /**
     * Gets a value indicating whether cancellation was signaled.
     */
    readonly signaled: boolean;
    /**
     * Gets the reason cancellation was signaled.
     */
    readonly reason: unknown;
    /**
     * Subscribes to notifications for when the object becomes signaled.
     */
    subscribe(onSignaled: () => void): CancelSubscription;
}
export interface CancelableCancelSignal extends CancelSignal {
    [Cancelable.cancelSignal](): CancelableCancelSignal;
}
/**
 * Represents an object that is a source for cancelation.
 */
export interface CancelableSource extends Cancelable {
    /**
     * Cancels the source, notifying the associated CancelSignal.
     */
    [CancelableSource.cancel](reason?: unknown): void;
}
export declare namespace CancelableSource {
    export import cancelSignal = Cancelable.cancelSignal;
    const cancel: unique symbol;
    const name = "CancelableSource";
    /**
     * Determines whether a value is a `CancelableSource` object.
     */
    function hasInstance(value: unknown): value is CancelableSource;
}
export {};
//# sourceMappingURL=index.d.ts.map