"use strict";
/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   THIRD PARTY LICENSE NOTICE:

   CancelToken is derived from the implementation of CancellationToken in
   Promise Extensions for Javascript: https://github.com/rbuckton/prex

   Promise Extensions is licensed under the Apache 2.0 License:

   Promise Extensions for JavaScript
   Copyright (c) Microsoft Corporation

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CancelToken = exports.CancelSubscription = exports.CancelError = void 0;
const cancelable_1 = require("@esfx/cancelable");
const disposable_1 = require("@esfx/disposable");
const internal_linked_list_1 = function () {
    var module = { exports: {} };
    (function (module, exports, require) {
        "use strict";
        Object.defineProperty(exports, "__esModule", { value: true });
        function listCreate() {
            return { size: 0, head: null };
        }
        exports.listCreate = listCreate;
        function listAdd(list, value) {
            const node = { value, next: null, prev: null };
            if (list.head === null) {
                list.head = node.next = node.prev = node;
            }
            else {
                const tail = list.head.prev;
                if (!(tail === null || tail === void 0 ? void 0 : tail.next))
                    throw new Error("Illegal state");
                node.prev = tail;
                node.next = tail.next;
                tail.next = tail.next.prev = node;
            }
            list.size++;
            return node;
        }
        exports.listAdd = listAdd;
        function listRemove(list, node) {
            if (node.next === null || node.prev === null)
                return false;
            if (node.next === node) {
                list.head = null;
            }
            else {
                node.next.prev = node.prev;
                node.prev.next = node.next;
                if (list.head === node) {
                    list.head = node.next;
                }
            }
            node.next = node.prev = null;
            list.size--;
            return true;
        }
        exports.listRemove = listRemove;
    })(module, module.exports, null);
    return module.exports;
}();
var cancelable_2 = require("@esfx/cancelable");
Object.defineProperty(exports, "CancelError", { enumerable: true, get: function () { return cancelable_2.CancelError; } });
Object.defineProperty(exports, "CancelSubscription", { enumerable: true, get: function () { return cancelable_2.CancelSubscription; } });
let AbortControllerAbort;
let AbortControllerGetSignal;
let AbortSignalAddEventListener;
let AbortSignalRemoveEventListener;
let AbortSignalGetAborted;
let AbortSignalGetReason;
const kEventTarget = Symbol.for("nodejs.event_target");
const isNodeAbortSignal = typeof AbortSignal === "function" && AbortSignal[kEventTarget] === true;
const nodeEventTargetProperties = new Set(isNodeAbortSignal ? ["dispatchEvent", ...Object.getOwnPropertySymbols(EventTarget.prototype)] : []);
if (typeof AbortController === "function" && typeof AbortSignal === "function") {
    const uncurryThis = Function.prototype.bind.bind(Function.prototype.call);
    AbortControllerAbort = uncurryThis(AbortController.prototype.abort);
    AbortControllerGetSignal = uncurryThis(Object.getOwnPropertyDescriptor(AbortController.prototype, "signal").get);
    AbortSignalAddEventListener = uncurryThis(AbortSignal.prototype.addEventListener);
    AbortSignalRemoveEventListener = uncurryThis(AbortSignal.prototype.removeEventListener);
    AbortSignalGetAborted = uncurryThis(Object.getOwnPropertyDescriptor(AbortSignal.prototype, "aborted").get);
    AbortSignalGetReason = "reason" in AbortSignal.prototype ? uncurryThis(Object.getOwnPropertyDescriptor(AbortSignal.prototype, "reason").get) : undefined;
}
// #endregion DOM AbortController/AbortSignal compatibility
const emptySubscription = cancelable_1.Cancelable.none.subscribe(() => { });
const defaultReason = cancelable_1.Cancelable.canceled.reason;
class CancelState {
    constructor(links) {
        this._state = "unsignaled";
        this._pendingCancelAfters = [];
        this._links = links;
    }
    get state() {
        return this._state === "unsignaled" && this._links ? this._links.getLinkedState() : this._state;
    }
    get source() {
        var _b;
        return (_b = this._source) !== null && _b !== void 0 ? _b : (this._source = new CancelSource(this));
    }
    get token() {
        var _b;
        return (_b = this._token) !== null && _b !== void 0 ? _b : (this._token = CancelState.createCancelToken(this));
    }
    get reason() {
        return this._reason;
    }
    get abortController() {
        if (!this._abortController && typeof AbortController === "function") {
            this._abortController = new AbortController();
            if (this._state === "signaled") {
                AbortControllerAbort(this._abortController, this._reason);
            }
        }
        return this._abortController;
    }
    get abortSignal() {
        const abortController = this.abortController;
        return abortController ? AbortControllerGetSignal(abortController) : undefined;
    }
    static race(signals) {
        const subscriptions = [];
        const cancelState = new CancelState({
            getLinkedState() {
                return signals.some(isSignaled) ? "signaled" :
                    signals.some(canBeSignaled) ? "unsignaled" :
                        "closed";
            },
            unlink() {
                if (subscriptions.length > 0) {
                    for (const subscription of subscriptions.splice(0, subscriptions.length)) {
                        subscription.unsubscribe();
                    }
                }
            }
        });
        for (const signal of signals) {
            let signaled = false;
            const onSignaled = () => {
                var _b;
                if (!signaled) {
                    signaled = true;
                    cancelState.cancel((_b = signal.reason) !== null && _b !== void 0 ? _b : defaultReason);
                }
            };
            if (signal.signaled) {
                onSignaled();
                break;
            }
            else {
                subscriptions.push(signal.subscribe(onSignaled));
            }
        }
        return cancelState;
    }
    cancel(reason) {
        if (this._state !== "unsignaled")
            return;
        const links = this._links;
        const subscriptions = this._subscriptions;
        const abortController = this._abortController;
        const pendingCancelAfters = this._pendingCancelAfters.splice(0, this._pendingCancelAfters.length);
        this._state = "signaled";
        this._reason = reason;
        this._links = undefined;
        this._subscriptions = undefined;
        this._abortController = undefined;
        for (const timeout of pendingCancelAfters) {
            clearTimeout(timeout);
        }
        try {
            links === null || links === void 0 ? void 0 : links.unlink();
            if (subscriptions) {
                const callbacks = [];
                while (subscriptions.head) {
                    // The registration for each callback holds onto the node, the node holds onto the
                    // list, and the list holds all other nodes and callbacks. By removing the node from
                    // the list, the GC can collect any otherwise unreachable nodes.
                    const head = subscriptions.head;
                    if ((0, internal_linked_list_1.listRemove)(subscriptions, head)) {
                        callbacks.push(head.value);
                        head.value = undefined;
                    }
                }
                for (const callback of callbacks) {
                    executeCallback(callback);
                }
            }
        }
        finally {
            if (abortController) {
                AbortControllerAbort(abortController, this._reason);
            }
        }
    }
    cancelAfter(ms, reason) {
        if (this._state === "unsignaled") {
            this._pendingCancelAfters.push(setTimeout(() => this.cancel(reason), ms));
        }
    }
    close() {
        if (this._state !== "unsignaled")
            return;
        const currentLinks = this._links;
        const currentSubscriptions = this._subscriptions;
        const pendingCancelAfters = this._pendingCancelAfters.splice(0, this._pendingCancelAfters.length);
        this._state = "closed";
        this._reason = undefined;
        this._links = undefined;
        this._subscriptions = undefined;
        this._abortController = undefined;
        for (const timeout of pendingCancelAfters) {
            clearTimeout(timeout);
        }
        currentLinks === null || currentLinks === void 0 ? void 0 : currentLinks.unlink();
        if (currentSubscriptions) {
            while (currentSubscriptions.head) {
                const head = currentSubscriptions.head;
                if ((0, internal_linked_list_1.listRemove)(currentSubscriptions, head)) {
                    head.value = undefined;
                }
            }
        }
    }
    subscribe(onSignaled) {
        var _b;
        if (this.state !== "unsignaled") {
            throw new Error("Illegal state");
        }
        const list = (_b = this._subscriptions) !== null && _b !== void 0 ? _b : (this._subscriptions = (0, internal_linked_list_1.listCreate)());
        return createCancelSubscription(list, (0, internal_linked_list_1.listAdd)(list, onSignaled));
    }
}
_a = CancelState;
(() => {
    _a.closed = new CancelState();
    _a.closed.close();
    _a.canceled = new CancelState();
    _a.canceled.cancel(new cancelable_1.CancelError());
})();
const weakCancelSourceState = new WeakMap();
const CancelSource = (_b = class {
        constructor(cancel) {
            weakCancelSourceState.set(this, cancel);
            Object.freeze(this);
        }
        get token() {
            const cancel = weakCancelSourceState.get(this);
            if (!cancel)
                throw new TypeError("Value of 'this' must be of type CancelSource");
            return cancel.token;
        }
        cancel(reason) {
            const cancel = weakCancelSourceState.get(this);
            if (!cancel)
                throw new TypeError("Value of 'this' must be of type CancelSource");
            if (cancel.state === "unsignaled") {
                cancel.cancel(reason !== null && reason !== void 0 ? reason : new cancelable_1.CancelError());
            }
        }
        cancelAfter(timeout, reason) {
            if (!(isFinite(timeout) && timeout >= 0))
                throw new RangeError("Argument out of range: timeout");
            const cancel = weakCancelSourceState.get(this);
            if (!cancel)
                throw new TypeError("Value of 'this' must be of type CancelSource");
            if (cancel.state === "unsignaled") {
                cancel.cancelAfter(timeout, reason !== null && reason !== void 0 ? reason : new cancelable_1.CancelError("Operation timed out"));
            }
        }
        close() {
            const cancel = weakCancelSourceState.get(this);
            if (!cancel)
                throw new TypeError("Value of 'this' must be of type CancelSource");
            cancel.close();
        }
        [cancelable_1.CancelableSource.cancelSignal]() {
            return this.token;
        }
        [cancelable_1.CancelableSource.cancel](reason) {
            this.cancel(reason);
        }
        [disposable_1.Disposable.dispose]() {
            this.close();
        }
    },
    (() => {
        // Hide the constructor from the prototype
        Object.defineProperty(_b.prototype, "constructor", { ...Object.getOwnPropertyDescriptor(_b.prototype, "constructor"), value: Object });
        Object.defineProperty(_b.prototype, Symbol.toStringTag, { configurable: true, value: "CancelSource" });
    })(),
    _b);
const weakCancelTokenState = new WeakMap();
/**
 * Propagates notifications that operations should be canceled.
 */
class CancelToken {
    constructor() {
        throw new TypeError("Object not creatable.");
    }
    /**
     * Gets a value indicating whether the token is signaled.
     */
    get signaled() {
        const cancel = weakCancelTokenState.get(this);
        if (!cancel)
            throw new TypeError("Value of 'this' must be of type CancelToken");
        return cancel.state === "signaled";
    }
    /**
     * Gets the reason for cancellation.
     */
    get reason() {
        const cancel = weakCancelTokenState.get(this);
        if (!cancel)
            throw new TypeError("Value of 'this' must be of type CancelToken");
        return cancel.reason;
    }
    /**
     * Gets a value indicating whether the token can be signaled.
     */
    get canBeSignaled() {
        const cancel = weakCancelTokenState.get(this);
        if (!cancel)
            throw new TypeError("Value of 'this' must be of type CancelToken");
        return cancel.state !== "closed";
    }
    /**
     * Adapts a {@link CancelToken} from a cancelable.
     */
    static from(cancelable) {
        if (cancelable === cancelable_1.Cancelable.none || (cancelable === null || cancelable === undefined)) {
            return CancelToken.none;
        }
        if (cancelable === cancelable_1.Cancelable.canceled) {
            return CancelToken.canceled;
        }
        if (cancelable instanceof CancelSource) {
            return cancelable.token;
        }
        if (cancelable instanceof CancelToken) {
            return cancelable;
        }
        if (typeof AbortSignal === "function" && cancelable instanceof AbortSignal) {
            return getCancelTokenFromAbortSignal(cancelable);
        }
        if (cancelable_1.Cancelable.hasInstance(cancelable)) {
            return getCancelTokenFromCancelable(cancelable);
        }
        throw new TypeError("Cancelable exepected: cancelable");
    }
    /**
     * Creates a new {@link CancelSource}.
     *
     * @param cancelables An optional iterable of {@link @esfx/cancelable!Cancelable} objects. If present,
     * the source becomes linked to the provided cancelables and will be canceled
     * when a linked cancelable is canceled.
     *
     * @remarks Calling {@link source} with `cancelables` is similar to {@link race}, except you can
     * individually cancel or close the resulting source. This can be better for memory or GC purposes,
     * since when the resulting source is canceled or closed it can be unlinked from the cancelables,
     * removing references from each cancelable to the resulting source which could otherwise prevent
     * garbage collection.
     */
    static source(cancelables) {
        if (cancelables === null || cancelables === undefined) {
            return new CancelState().source;
        }
        if (!((typeof cancelables === "object" && cancelables !== null || typeof cancelables === "function") && Symbol.iterator in cancelables)) {
            throw new TypeError("Object not iterable: cancelables");
        }
        const signals = getCancelSignalArrayFromCancelables(cancelables);
        if (!signals) {
            throw new TypeError("Cancelable element expected: cancelables");
        }
        if (!signals.some(canBeSignaled)) {
            return new CancelState().source;
        }
        return CancelState.race(signals).source;
    }
    /**
     * Returns a {@link CancelToken} that becomes signaled when any of the provided cancelables are signaled.
     *
     * @param cancelables An iterable of {@link @esfx/cancelable!Cancelable} objects.
     *
     * @remarks This is similar to calling {@link source} with the provided `cancelables`. In general,
     * calling {@link source} is preferred as it provides better resource management.
     */
    static race(cancelables) {
        if (!((typeof cancelables === "object" && cancelables !== null || typeof cancelables === "function") && Symbol.iterator in cancelables)) {
            throw new TypeError("Object not iterable: cancelables");
        }
        const signals = getCancelSignalArrayFromCancelables(cancelables);
        if (!signals) {
            throw new TypeError("Cancelable element expected: cancelables");
        }
        if (!signals.some(canBeSignaled)) {
            return CancelToken.none;
        }
        return CancelState.race(signals).token;
    }
    /**
     * Returns a {@link CancelToken} that becomes signaled when all of the provided cancelables are signaled.
     *
     * @param cancelables An iterable of {@link @esfx/cancelable!Cancelable} objects.
     */
    static all(cancelables) {
        if (!((typeof cancelables === "object" && cancelables !== null || typeof cancelables === "function") && Symbol.iterator in cancelables)) {
            throw new TypeError("Object not iterable: cancelables");
        }
        const signals = getCancelSignalArrayFromCancelables(cancelables);
        if (!signals) {
            throw new TypeError("Cancelable element expected: cancelables");
        }
        if (!signals.some(canBeSignaled)) {
            return CancelToken.none;
        }
        const reasons = [];
        const subscriptions = [];
        const cancelState = new CancelState({
            getLinkedState() {
                return signals.every(isSignaled) ? "signaled" :
                    signals.some(canBeSignaled) ? "unsignaled" :
                        "closed";
            },
            unlink() {
                if (subscriptions.length > 0) {
                    for (const subscription of subscriptions.splice(0, subscriptions.length)) {
                        subscription.unsubscribe();
                    }
                }
            }
        });
        let signalsRemaining = signals.length;
        for (let i = 0; i < signals.length; i++) {
            let signaled = false;
            const signal = signals[i];
            const onSignaled = () => {
                var _b;
                if (!signaled) {
                    signaled = true;
                    signalsRemaining--;
                    reasons[i] = (_b = signal.reason) !== null && _b !== void 0 ? _b : defaultReason;
                    if (signalsRemaining === 0) {
                        const error = new cancelable_1.CancelError();
                        if (!reasons.every(reason => reason === defaultReason)) {
                            Object.defineProperty(error, "cause", { enumerable: false, configurable: true, writable: true, value: new AggregateError(reasons) });
                        }
                        cancelState.cancel(error);
                    }
                }
            };
            if (signal.signaled) {
                onSignaled();
            }
            else {
                subscriptions.push(signal.subscribe(onSignaled));
            }
        }
        return cancelState.token;
    }
    /**
     * Gets a {@link CancelToken} that is already canceled with the provided reason.
     */
    static canceledWith(reason) {
        const cancelState = new CancelState();
        cancelState.cancel(reason !== null && reason !== void 0 ? reason : new cancelable_1.CancelError());
        return cancelState.token;
    }
    /**
     * Gets a {@link CancelToken} that will be canceled with the provided reason after a timeout has elapsed.
     */
    static timeout(ms, reason) {
        if (!(isFinite(ms) && ms >= 0))
            throw new RangeError("Argument out of range: ms");
        const cancelState = new CancelState();
        cancelState.cancelAfter(ms, reason !== null && reason !== void 0 ? reason : new cancelable_1.CancelError("Operation timed out"));
        return cancelState.token;
    }
    /**
     * Throws a {@link @esfx/cancelable!CancelError} if the token was signaled.
     */
    throwIfSignaled() {
        const cancel = weakCancelTokenState.get(this);
        if (!cancel)
            throw new TypeError("Value of 'this' must be of type CancelToken");
        if (cancel.state === "signaled") {
            throw cancel.reason;
        }
    }
    /**
     * Subscribes to notifications for when the object becomes signaled.
     */
    subscribe(onSignaled) {
        if (!(typeof onSignaled === "function"))
            throw new TypeError("Function expected: onSignaled");
        const cancel = weakCancelTokenState.get(this);
        if (!cancel)
            throw new TypeError("Value of 'this' must be of type CancelToken");
        if (cancel.state === "closed")
            return emptySubscription;
        if (cancel.state === "signaled")
            return onSignaled(), emptySubscription;
        return cancel.subscribe(onSignaled);
    }
    // #region Cancelable
    /** {@inheritDoc @esfx/cancelable!Cancelable#[Cancelable.cancelSignal]:member(1)} */
    [(_d = kEventTarget, cancelable_1.Cancelable.cancelSignal)]() {
        return this;
    }
}
exports.CancelToken = CancelToken;
_c = CancelToken;
CancelToken[_d] = isNodeAbortSignal; // treat this as a NodeJS event target
(() => {
    CancelState.createCancelToken = cancelState => {
        // If it is available, make the token an `AbortSignal` instance with a `CancelToken` prototype.
        // This gives the token the same internal state as an `AbortSignal` and allows the token to be
        // used in `fetch` and other DOM apis.
        const abortSignal = cancelState.abortSignal;
        let token;
        if (abortSignal) {
            token = Object.setPrototypeOf(abortSignal, CancelToken.prototype);
        }
        else {
            token = Object.create(CancelToken.prototype);
        }
        weakCancelTokenState.set(token, cancelState);
        Object.preventExtensions(token);
        return token;
    };
    Object.defineProperty(_c.prototype, Symbol.toStringTag, { configurable: true, value: "CancelToken" });
})();
CancelToken.none = CancelState.closed.token;
CancelToken.canceled = CancelState.canceled.token;
if (isNodeAbortSignal) {
    Object.setPrototypeOf(CancelToken.prototype, EventTarget.prototype);
}
// helpers
const weakCancelTokenFromCancelSignal = new WeakMap();
const weakCancelTokenFromAbortSignal = typeof AbortSignal === "function" ? new WeakMap() : undefined;
const weakCancelSignalFromAbortSignal = typeof AbortSignal === "function" ? new WeakMap() : undefined;
function getCancelSignalArrayFromCancelables(cancelables) {
    const signals = [];
    for (const cancelable of cancelables) {
        const signal = cancelable === null || cancelable === undefined ? CancelToken.none :
            cancelable instanceof CancelToken ? cancelable :
                typeof AbortSignal === "function" && cancelable instanceof AbortSignal ? getCancelSignalFromAbortSignal(cancelable) :
                    cancelable_1.Cancelable.hasInstance(cancelable) ? cancelable[cancelable_1.Cancelable.cancelSignal]() :
                        undefined;
        if (!signal) {
            return undefined;
        }
        signals.push(signal);
    }
    return signals;
}
function getCancelTokenFromCancelSignal(cancelSignal) {
    if (cancelSignal instanceof CancelToken) {
        return cancelSignal;
    }
    let cancelToken = weakCancelTokenFromCancelSignal === null || weakCancelTokenFromCancelSignal === void 0 ? void 0 : weakCancelTokenFromCancelSignal.get(cancelSignal);
    if (!cancelToken) {
        let subscription;
        const cancelState = new CancelState({
            getLinkedState() {
                return isSignaled(cancelSignal) ? "signaled" :
                    canBeSignaled(cancelSignal) ? "unsignaled" :
                        "closed";
            },
            unlink() {
                if (subscription) {
                    subscription.unsubscribe();
                    subscription = undefined;
                }
            }
        });
        subscription = cancelSignal.subscribe(() => {
            var _b;
            cancelState.cancel((_b = cancelSignal.reason) !== null && _b !== void 0 ? _b : defaultReason);
        });
        cancelToken = cancelState.token;
        weakCancelTokenFromCancelSignal === null || weakCancelTokenFromCancelSignal === void 0 ? void 0 : weakCancelTokenFromCancelSignal.set(cancelSignal, cancelToken);
    }
    return cancelToken;
}
function getCancelTokenFromCancelable(cancelable) {
    if (cancelable instanceof CancelToken) {
        return cancelable;
    }
    return getCancelTokenFromCancelSignal(cancelable[cancelable_1.Cancelable.cancelSignal]());
}
function getCancelSignalFromAbortSignal(abortSignal) {
    if (abortSignal instanceof CancelToken) {
        return abortSignal;
    }
    let cancelSignal = weakCancelSignalFromAbortSignal === null || weakCancelSignalFromAbortSignal === void 0 ? void 0 : weakCancelSignalFromAbortSignal.get(abortSignal);
    if (!cancelSignal) {
        cancelSignal = {
            get signaled() {
                return AbortSignalGetAborted(abortSignal);
            },
            get reason() {
                var _b;
                return (_b = AbortSignalGetReason === null || AbortSignalGetReason === void 0 ? void 0 : AbortSignalGetReason(abortSignal)) !== null && _b !== void 0 ? _b : defaultReason;
            },
            subscribe(onSignaled) {
                if (!(typeof onSignaled === "function"))
                    throw new TypeError("Function expected: onSignaled");
                let subscribed = true;
                const onAbort = () => {
                    if (subscribed) {
                        subscribed = false;
                        AbortSignalRemoveEventListener(abortSignal, "abort", onAbort);
                        onSignaled();
                        onSignaled = undefined;
                    }
                };
                AbortSignalAddEventListener(abortSignal, "abort", onAbort);
                return cancelable_1.CancelSubscription.create(() => {
                    if (subscribed) {
                        subscribed = false;
                        AbortSignalRemoveEventListener(abortSignal, "abort", onAbort);
                        onSignaled = undefined;
                    }
                });
            }
        };
        weakCancelSignalFromAbortSignal === null || weakCancelSignalFromAbortSignal === void 0 ? void 0 : weakCancelSignalFromAbortSignal.set(abortSignal, cancelSignal);
    }
    return cancelSignal;
}
function getCancelTokenFromAbortSignal(abortSignal) {
    if (abortSignal instanceof CancelToken) {
        return abortSignal;
    }
    let cancelToken = weakCancelTokenFromAbortSignal === null || weakCancelTokenFromAbortSignal === void 0 ? void 0 : weakCancelTokenFromAbortSignal.get(abortSignal);
    if (!cancelToken) {
        cancelToken = getCancelTokenFromCancelSignal(getCancelSignalFromAbortSignal(abortSignal));
        weakCancelTokenFromAbortSignal === null || weakCancelTokenFromAbortSignal === void 0 ? void 0 : weakCancelTokenFromAbortSignal.set(abortSignal, cancelToken);
    }
    return cancelToken;
}
function executeCallback(callback) {
    try {
        callback();
    }
    catch (e) {
        // HostReportError(e);
        setImmediate(() => { throw e; });
    }
}
function isSignaled(signal) {
    return signal.signaled;
}
function canBeSignaled(signal) {
    return signal !== cancelable_1.Cancelable.none && (!(signal instanceof CancelToken) || signal.canBeSignaled);
}
function createCancelSubscription(list, node) {
    return cancelable_1.CancelSubscription.create(() => {
        if (list && node && (0, internal_linked_list_1.listRemove)(list, node)) {
            node.value = undefined;
            node = undefined;
            list = undefined;
        }
    });
}
//# sourceMappingURL=index.js.map