/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
/**
 * Indicates an object that has resources that can be explicitly disposed.
 */
export interface Disposable {
    /**
     * Dispose this object's resources.
     */
    [Disposable.dispose](): void;
}
/**
 * Indicates an object that has resources that can be explicitly disposed.
 */
export declare class Disposable {
    /**
     * Creates a `Disposable` wrapper around a callback used to dispose of a resource.
     * @deprecated Use `DisposableStack` or `{ [Disposable.dispose]() { ... } }` instead.
     */
    constructor(dispose: () => void);
}
/**
 * Indicates an object that has resources that can be explicitly disposed.
 */
export declare namespace Disposable {
    /**
     * A well-known symbol used to define an explicit resource disposal method on an object.
     *
     * NOTE: Uses `Symbol.dispose` if present.
     */
    const dispose: unique symbol;
    /**
     * Emulate `using const` using `for..of`.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     *
     * @example
     * ```ts
     * // with `using const` (proposed)
     * {
     *   ...
     *   using const x = expr, y = expr;
     *   ...
     * }
     *
     * // with `Disposable.scope()`:
     * for (const { using, fail } of Disposable.scope()) {
     *   try {
     *     ...
     *     const x = using(expr), y = using(expr);
     *     ...
     *   }
     *   catch (e) {
     *     fail(e);
     *   }
     * }
     * ```
     */
    function scope(): Generator<DisposableScope, void, undefined>;
    /**
     * Yields each disposable in the iterable, disposing it when the generator resumes.
     *
     * This emulates `for (using const x of expr)`.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     *
     * @example
     * ```ts
     * // with `using const` (proposed)
     * for (using const x of expr) {
     *   ...
     * }
     *
     * // with `Disposable.usingEach()`:
     * for (const x of Disposable.usingEach(expr)) {
     *   ...
     * }
     * ```
     */
    function usingEach(iterable: Iterable<Disposable | null | undefined>): Generator<Disposable | null | undefined, void, unknown>;
    /**
     * Creates a `Disposable` wrapper around a callback used to dispose of a resource.
     *
     * NOTE: This is not spec-compliant and will not be standardized. It is preferred to use a `DisposableStack`
     * or to implement `Disposable.dispose` yourself instead.
     */
    function create(dispose: () => void): Disposable;
    /**
     * Determines whether a value is Disposable.
     *
     * NOTE: This is not spec-compliant and will not be standardized.
     */
    function hasInstance(value: unknown): value is Disposable;
}
/**
 * Used to aproximate `using const` via `for..of`. See {@link Disposable.scope}.
 *
 * NOTE: This is not spec-compliant and will not be standardized.
 */
export interface DisposableScope {
    /**
     * Tracks a resource to be disposed at the end of a `for..of` statement. See {@link Disposable.scope}.
     */
    using<T extends Disposable | null | undefined>(value: T): T;
    /**
     * Tracks an exception from the body of a `for..of` statement. See {@link Disposable.scope}.
     */
    fail(error: unknown): void;
}
//# sourceMappingURL=disposable.d.ts.map