"use strict";
/*!
   Copyright 2019 Ron Buckton

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CancelableSource = exports.Cancelable = exports.CancelSubscription = exports.CancelError = void 0;
const disposable_1 = require("@esfx/disposable");
class CancelError extends Error {
    constructor(message = "Operation was canceled", ...args) {
        super(message, ...args);
    }
}
exports.CancelError = CancelError;
_a = CancelError;
(() => {
    Object.defineProperty(_a.prototype, "name", { configurable: true, writable: true, value: "CancelError" });
})();
var CancelSubscription;
(function (CancelSubscription) {
    const CancelSubscriptionPrototype = {
        [disposable_1.Disposable.dispose]() {
            this.unsubscribe();
        }
    };
    Object.defineProperty(CancelSubscriptionPrototype, Symbol.toStringTag, { configurable: true, value: "CancelSubscription" });
    /**
     * Creates a `CancelSubscription` object for an `unsubscribe` callback.
     * @param unsubscribe The callback to execute when the `unsubscribe()` method is called.
     */
    function create(unsubscribe) {
        if (!(typeof unsubscribe === "function"))
            throw new TypeError("Function expected: unsubscribe");
        return Object.setPrototypeOf({
            unsubscribe() {
                unsubscribe();
            },
        }, CancelSubscriptionPrototype);
    }
    CancelSubscription.create = create;
})(CancelSubscription = exports.CancelSubscription || (exports.CancelSubscription = {}));
var Cancelable;
(function (Cancelable) {
    // #region Cancelable
    /**
     * A well-known symbol used to define a method to retrieve the `CancelSignal` for an object.
     */
    Cancelable.cancelSignal = Symbol.for("@esfx/cancelable:Cancelable.cancelSignal");
    // #endregion Cancelable
    const CancelSignalPrototype = {
        [Cancelable.cancelSignal]() {
            return this;
        }
    };
    Object.defineProperty(CancelSignalPrototype, Symbol.toStringTag, { configurable: true, value: "CancelSignal" });
    const emptySubscription = CancelSubscription.create(() => { });
    Object.freeze(emptySubscription);
    const canceledReason = new CancelError();
    Object.freeze(canceledReason);
    /**
     * A `Cancelable` that is already signaled.
     */
    Cancelable.canceled = Object.setPrototypeOf({
        get signaled() {
            return true;
        },
        get reason() {
            return canceledReason;
        },
        subscribe(onSignaled) {
            if (!(typeof onSignaled === "function"))
                throw new TypeError("Function expected: onSignaled");
            onSignaled();
            return emptySubscription;
        }
    }, CancelSignalPrototype);
    Object.freeze(Cancelable.canceled);
    /**
     * A `Cancelable` that can never be signaled.
     */
    Cancelable.none = Object.setPrototypeOf({
        get signaled() {
            return false;
        },
        get reason() {
            return undefined;
        },
        subscribe(onSignaled) {
            if (!(typeof onSignaled === "function"))
                throw new TypeError("Function expected: onSignaled");
            return emptySubscription;
        },
    }, CancelSignalPrototype);
    Object.freeze(Cancelable.none);
    /**
     * Determines whether `cancelable` is in the signaled state.
     */
    function isSignaled(cancelable) {
        if (cancelable === null || cancelable === undefined)
            return false;
        if (!hasInstance(cancelable))
            throw new TypeError("Cancelable expected: cancelable");
        if (cancelable === Cancelable.canceled)
            return true;
        if (cancelable === Cancelable.none)
            return false;
        return cancelable[Cancelable.cancelSignal]().signaled;
    }
    Cancelable.isSignaled = isSignaled;
    /**
     * Gets the reason for cancelation.
     */
    function getReason(cancelable) {
        if (cancelable === null || cancelable === undefined)
            return undefined;
        if (!hasInstance(cancelable))
            throw new TypeError("Cancelable expected: cancelable");
        if (cancelable === Cancelable.canceled)
            return canceledReason;
        if (cancelable === Cancelable.none)
            return undefined;
        return cancelable[Cancelable.cancelSignal]().reason;
    }
    Cancelable.getReason = getReason;
    /**
     * Throws a `CancelError` exception if the provided `cancelable` is in the signaled state.
     */
    function throwIfSignaled(cancelable) {
        var _b;
        if (isSignaled(cancelable)) {
            throw (_b = getReason(cancelable)) !== null && _b !== void 0 ? _b : new CancelError();
        }
    }
    Cancelable.throwIfSignaled = throwIfSignaled;
    /**
     * Subscribes to be notified when a `cancelable` becomes signaled.
     */
    function subscribe(cancelable, onSignaled) {
        if (cancelable === null || cancelable === undefined)
            return emptySubscription;
        if (!hasInstance(cancelable))
            throw new TypeError("Cancelable expected: cancelable");
        if (cancelable === Cancelable.canceled)
            return onSignaled(), emptySubscription;
        if (cancelable === Cancelable.none)
            return emptySubscription;
        return cancelable[Cancelable.cancelSignal]().subscribe(onSignaled);
    }
    Cancelable.subscribe = subscribe;
    Cancelable.name = "Cancelable";
    /**
     * Determines whether a value is a `Cancelable` object.
     */
    function hasInstance(value) {
        return (typeof value === "object" && value !== null || typeof value === "function") && Cancelable.cancelSignal in value;
    }
    Cancelable.hasInstance = hasInstance;
})(Cancelable = exports.Cancelable || (exports.Cancelable = {}));
var CancelableSource;
(function (CancelableSource) {
    // #region Cancelable
    CancelableSource.cancelSignal = Cancelable.cancelSignal;
    // #endregion Cancelable
    // #region CancelableSource
    CancelableSource.cancel = Symbol.for("@esfx/cancelable:CancelableSource.cancel");
    // #endregion CancelableSource
    CancelableSource.name = "CancelableSource";
    /**
     * Determines whether a value is a `CancelableSource` object.
     */
    function hasInstance(value) {
        return Cancelable.hasInstance(value)
            && CancelableSource.cancel in value;
    }
    CancelableSource.hasInstance = hasInstance;
})(CancelableSource = exports.CancelableSource || (exports.CancelableSource = {}));
//# sourceMappingURL=index.js.map